/*
 * BSD License
 *  ( http://www.opensource.org/licenses/bsd-license.html )
 *
 * Software copyright (c) 2002, k` (agggka@hotmail.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution
 *  3. The name of the author may not be used to endorse or promote
 *     products derived from this software without specific prior
 *     written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * 
 * Created on 7 sept. 2003
 */

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

package org.skreel.k.isola.core;


/* # imports - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */


/* # classe GameState  - - - - - - - - - - - - - - - - - - - - - - - - - - - */
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

/**
 * Description de l'etat du jeu a un instant donne, soit le plateau de jeu
 *  et les positions des joueurs. L'ajout d'un contexte permet une
 *  description fine d'un etat.
 *
 * @author k
 */
public final class GameState
{
 /* # proprietes  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
 /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

 /**
  * Plateau de jeu.
  */
  byte[]      gameBoard;


 /**
  * Positions des joueurs.
  */
  GamePoint[] positions;


 /**
  * Contexte du jeu.
  */
  int         gameContxt;



 /* # constructeur  - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
 /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

 /**
  * Construit un etat de jeu, un plateau vierge, deux joueurs places aux
  *  positions de depart.
  */
  public GameState()
  { this.gameBoard= new byte[Game.BOARD_ROWS];
    this.positions= new GamePoint[2];

    this.positions[Game.PLAYER_ONE]= new GamePoint();
    this.positions[Game.PLAYER_TWO]= new GamePoint();

    this.reset();
  }



 /* # getter / setter - - - - - - - - - - - - - - - - - - - - - - - - - - - */
 /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

 /**
  * Recupere la plateau de jeu, sous forme d'un tableau d'entiers a deux
  *  dimensions. La valeur 1 indique une case detruite.
  *
  * @return                 l'etat du plateau de jeu
  */
  public final int[][] getGameBoard()
  { int[][] retour= new int[Game.BOARD_ROWS][Game.BOARD_COLS];

   /* ligne 1 - - - - - - - - - - - - - - - - - - - - - - - */
    retour[0][0]= (this.gameBoard[0]>>7) & 0x01;
    retour[0][1]= (this.gameBoard[0]>>6) & 0x01;
    retour[0][2]= (this.gameBoard[0]>>5) & 0x01;
    retour[0][3]= (this.gameBoard[0]>>4) & 0x01;
    retour[0][4]= (this.gameBoard[0]>>3) & 0x01;
    retour[0][5]= (this.gameBoard[0]>>2) & 0x01;
    retour[0][6]= (this.gameBoard[0]>>1) & 0x01;
    retour[0][7]=  this.gameBoard[0]     & 0x01;
   /* ligne 2 - - - - - - - - - - - - - - - - - - - - - - - */
    retour[1][0]= (this.gameBoard[1]>>7) & 0x01;
    retour[1][1]= (this.gameBoard[1]>>6) & 0x01;
    retour[1][2]= (this.gameBoard[1]>>5) & 0x01;
    retour[1][3]= (this.gameBoard[1]>>4) & 0x01;
    retour[1][4]= (this.gameBoard[1]>>3) & 0x01;
    retour[1][5]= (this.gameBoard[1]>>2) & 0x01;
    retour[1][6]= (this.gameBoard[1]>>1) & 0x01;
    retour[1][7]=  this.gameBoard[1]     & 0x01;
   /* ligne 3 - - - - - - - - - - - - - - - - - - - - - - - */
    retour[2][0]= (this.gameBoard[2]>>7) & 0x01;
    retour[2][1]= (this.gameBoard[2]>>6) & 0x01;
    retour[2][2]= (this.gameBoard[2]>>5) & 0x01;
    retour[2][3]= (this.gameBoard[2]>>4) & 0x01;
    retour[2][4]= (this.gameBoard[2]>>3) & 0x01;
    retour[2][5]= (this.gameBoard[2]>>2) & 0x01;
    retour[2][6]= (this.gameBoard[2]>>1) & 0x01;
    retour[2][7]=  this.gameBoard[2]     & 0x01;
   /* ligne 4 - - - - - - - - - - - - - - - - - - - - - - - */
    retour[3][0]= (this.gameBoard[3]>>7) & 0x01;
    retour[3][1]= (this.gameBoard[3]>>6) & 0x01;
    retour[3][2]= (this.gameBoard[3]>>5) & 0x01;
    retour[3][3]= (this.gameBoard[3]>>4) & 0x01;
    retour[3][4]= (this.gameBoard[3]>>3) & 0x01;
    retour[3][5]= (this.gameBoard[3]>>2) & 0x01;
    retour[3][6]= (this.gameBoard[3]>>1) & 0x01;
    retour[3][7]=  this.gameBoard[3]     & 0x01;
   /* ligne 5 - - - - - - - - - - - - - - - - - - - - - - - */
    retour[4][0]= (this.gameBoard[4]>>7) & 0x01;
    retour[4][1]= (this.gameBoard[4]>>6) & 0x01;
    retour[4][2]= (this.gameBoard[4]>>5) & 0x01;
    retour[4][3]= (this.gameBoard[4]>>4) & 0x01;
    retour[4][4]= (this.gameBoard[4]>>3) & 0x01;
    retour[4][5]= (this.gameBoard[4]>>2) & 0x01;
    retour[4][6]= (this.gameBoard[4]>>1) & 0x01;
    retour[4][7]=  this.gameBoard[4]     & 0x01;
   /* ligne 6 - - - - - - - - - - - - - - - - - - - - - - - */
    retour[5][0]= (this.gameBoard[5]>>7) & 0x01;
    retour[5][1]= (this.gameBoard[5]>>6) & 0x01;
    retour[5][2]= (this.gameBoard[5]>>5) & 0x01;
    retour[5][3]= (this.gameBoard[5]>>4) & 0x01;
    retour[5][4]= (this.gameBoard[5]>>3) & 0x01;
    retour[5][5]= (this.gameBoard[5]>>2) & 0x01;
    retour[5][6]= (this.gameBoard[5]>>1) & 0x01;
    retour[5][7]=  this.gameBoard[5]     & 0x01;

    return (retour);
  }


 /**
  * Retourne les positions des deux joueurs sous la forme d'un tableau de
  *  deux points.
  * 
  * @return                 les positions courantes des deux joueurs
  */
  public final GamePoint[] getPositions()
  { return (this.positions); }


 /**
  * Retourne le contexte courant.
  *
  * @return                 contexte de jeu.
  */
  public final int getGameContext()
  { return (this.gameContxt); }


 /**
  * Retourne la position du joueur player, sous forme d'un objet GamePoint.
  *
  * @param playerId         identifiant du joueur
  * @return                 la position courante du joueur considere
  */
  final GamePoint getPlayerPosition(int playerId)
  { return (this.positions[playerId & 0x01]); }


 /**
  * Recupere les positions des cases directement adjacentes a la
  *  position (x,y). Un bit mit a 1 signifie case pratiquable.
  * 
  * @param x                abscisse du point a considerer
  * @param y                ordonnee du point a considerer
  * @return                 mot binaire
  */
  public final int getLayer1(int x, int y)
  { int retour= 0x00FF;
    int masque= GameBoard.getLayer1(x, y);
    int machin= (7-x);

   // les cases occupees par les joueurs sont vues comme impraticables
    this.switchPoint( this.positions[Game.PLAYER_ONE] );
    this.switchPoint( this.positions[Game.PLAYER_TWO] );

   // attention x correspond au bit (7-x) en partant de la droite !
    switch (x)
    { case 0 :
        switch (y)
        { case 0 : // { 0, 0 }
            retour&= ((this.gameBoard[0]>>>3) | 0xF7);
            retour&= ((this.gameBoard[1]>>>6) | 0xFC);
            break;
          case 5 : // { 0, 5 }
            retour&= ((this.gameBoard[4]>>>1) | 0x9F);
            retour&= ((this.gameBoard[5]>>>3) | 0xF7);
            break;
          default : // { 0, ? }
            retour&= ((this.gameBoard[y-1]>>>1) | 0x9F);
            retour&= ((this.gameBoard[y  ]>>>3) | 0xF7);
            retour&= ((this.gameBoard[y+1]>>>6) | 0xFC);
        }// switch (y)
        break;

      case 7 :
        switch (y)
        { case 0 : // { 0, 0 }
            retour&= ((this.gameBoard[0]<<3) | 0xEF);
            retour&= ((this.gameBoard[1]<<1) | 0xF9);
            break;
          case 5 : // { 0, 5 }
            retour&= ((this.gameBoard[4]<<6) | 0x3F);
            retour&= ((this.gameBoard[5]<<3) | 0xEF);
            break;
          default : // { 0, ? }
            retour&= ((this.gameBoard[y-1]<<6) | 0x3F);
            retour&= ((this.gameBoard[y  ]<<3) | 0xEF);
            retour&= ((this.gameBoard[y+1]<<1) | 0xF9);
        }// switch (y)
        break;

      default :
        switch (y)
        { case 0 : // { ?, 0 }
            retour&= (((this.gameBoard[0]>> machin   )<<3) | 0xEF);
            retour&= (((this.gameBoard[0]>>(machin-1))<<3) | 0xF7);
            retour&= ( (this.gameBoard[1]>>(machin-1))     | 0xF8);
            break;
          case 5 : // { ?, 5 }
            retour&= (((this.gameBoard[4]>>(machin-1))<<5) | 0x1F);
            retour&= (((this.gameBoard[5]>> machin   )<<3) | 0xEF);
            retour&= (((this.gameBoard[5]>>(machin-1))<<3) | 0xF7);
            break;
          default : // { ?, ? }
            retour&= (((this.gameBoard[y-1]>>>(machin-1))<<5) | 0x1F);
            retour&= (((this.gameBoard[y  ]>>> machin   )<<3) | 0xEF);
            retour&= (((this.gameBoard[y  ]>>>(machin-1))<<3) | 0xF7);
            retour&= ( (this.gameBoard[y+1]>>>(machin-1))     | 0xF8);
        }// switch (y)
    }// switch (x)

   // on retablit l'etat des cases occupees par les joueurs
    this.switchPoint( this.positions[Game.PLAYER_ONE] );
    this.switchPoint( this.positions[Game.PLAYER_TWO] );

    return ((~retour) & masque);
  }


 /**
  * Recupere les positions des cases directement adjacentes au point p.
  * 
  * @param p                position a considerer
  * @return                 mot binaire
  */
  public final int getLayer1(GamePoint p)
  { return (this.getLayer1(p.x, p.y)); }


 /**
  * Recupere les positions des cases situees a un mouvement d'ecart de
  *  la position (x,y). Un bit mit a 1 signifie case pratiquable.
  * 
  * @param x                abscisse du point a considerer
  * @param y                ordonnee du point a considerer
  * @return                 mot binaire
  */
  public final int getLayer2(int x, int y)
  { int retour= 0xFFFF;
    int masque= GameBoard.getLayer2(x, y);
    int machin= (7-x);

   // les cases occupees par les joueurs sont vues comme impraticables
    this.switchPoint( this.positions[Game.PLAYER_ONE] );
    this.switchPoint( this.positions[Game.PLAYER_TWO] );

   // attention x correspond au bit (7-x) en partant de la droite !
    switch (x)
    { case  0 :
        switch (y)
        { case 0 : // { 0, 0 }
            retour&= ((this.gameBoard[0]<<2) | 0xFF7F);
            retour&= ( this.gameBoard[1]     | 0xFFDF);
            retour&= ((this.gameBoard[2]>>5) | 0xFFF8);
            break;
          case 1 : // { 0, 1 }
            retour&= ((this.gameBoard[0]<<4) | 0xFDFF);
            retour&= ((this.gameBoard[1]<<2) | 0xFF7F);
            retour&= ( this.gameBoard[2]     | 0xFFDF);
            retour&= ((this.gameBoard[3]>>5) | 0xFFF8);
            break;
          case 4 : // { 0, 4 }
            retour&= ((this.gameBoard[2]<<6) | 0xC7FF);
            retour&= ((this.gameBoard[3]<<4) | 0xFDFF);
            retour&= ((this.gameBoard[4]<<2) | 0xFF7F);
            retour&= ( this.gameBoard[5]     | 0xFFDF);
            break;
          case 5 : // { 0, 5 }
            retour&= ((this.gameBoard[3]<<6) | 0xC7FF);
            retour&= ((this.gameBoard[4]<<4) | 0xFDFF);
            retour&= ((this.gameBoard[5]<<2) | 0xFF7F);
            break;
          default: // { 0, ? }
            retour&= ((this.gameBoard[y-2]<<6) | 0xC7FF);
            retour&= ((this.gameBoard[y-1]<<4) | 0xFDFF);
            retour&= ((this.gameBoard[y  ]<<2) | 0xFF7F);
            retour&= ( this.gameBoard[y+1]     | 0xFFDF);
            retour&= ((this.gameBoard[y+2]>>5) | 0xFFF8);
         }// switch (y)
         break;

       case  1 :
         switch (y)
         { case 0 : // { 1, 0 }
             retour&= ((this.gameBoard[0]<<3) | 0xFF7F);
             retour&= ((this.gameBoard[1]<<1) | 0xFFDF);
             retour&= ((this.gameBoard[2]>>4) | 0xFFF0);
             break;
           case 1 : // { 1, 1 }
             retour&= ((this.gameBoard[0]<<5) | 0xFDFF);
             retour&= ((this.gameBoard[1]<<3) | 0xFF7F);
             retour&= ((this.gameBoard[2]<<1) | 0xFFDF);
             retour&= ((this.gameBoard[3]>>4) | 0xFFF0);
             break;
           case 4 : // { 1, 4 }
             retour&= ((this.gameBoard[2]<<7) | 0x87FF);
             retour&= ((this.gameBoard[3]<<5) | 0xFDFF);
             retour&= ((this.gameBoard[4]<<3) | 0xFF7F);
             retour&= ((this.gameBoard[5]<<1) | 0xFFDF);
             break;
           case 5 : // { 1, 5 }
             retour&= ((this.gameBoard[3]<<7) | 0x87FF);
             retour&= ((this.gameBoard[4]<<5) | 0xFDFF);
             retour&= ((this.gameBoard[5]<<3) | 0xFF7F);
             break;
           default: // { 1, ? }
             retour&= ((this.gameBoard[y-2]<<7) | 0x87FF);
             retour&= ((this.gameBoard[y-1]<<5) | 0xFDFF);
             retour&= ((this.gameBoard[y  ]<<3) | 0xFF7F);
             retour&= ((this.gameBoard[y+1]<<1) | 0xFFDF);
             retour&= ((this.gameBoard[y+2]>>4) | 0xFFF0);
         }// switch (y)
         break;

       case  6 :
         switch (y)
         { case 0 : // { 6, 0 }
             retour&= ((this.gameBoard[0]<< 5) | 0xFEFF);
             retour&= ((this.gameBoard[1]<< 3) | 0xFFBF);
             retour&= ((this.gameBoard[2]<< 1) | 0xFFE1);
             break;
           case 1 : // { 6, 1 }
             retour&= ((this.gameBoard[0]<< 7) | 0xFBFF);
             retour&= ((this.gameBoard[1]<< 5) | 0xFEFF);
             retour&= ((this.gameBoard[2]<< 3) | 0xFFBF);
             retour&= ((this.gameBoard[3]<< 1) | 0xFFE1);
             break;
           case 4 : // { 6, 4 }
             retour&= ((this.gameBoard[2]<<12) | 0x0FFF);
             retour&= ((this.gameBoard[3]<< 7) | 0xFBFF);
             retour&= ((this.gameBoard[4]<< 5) | 0xFEFF);
             retour&= ((this.gameBoard[5]<< 3) | 0xFFBF);
             break;
           case 5 : // { 6, 5 }
             retour&= ((this.gameBoard[3]<<12) | 0x0FFF);
             retour&= ((this.gameBoard[4]<< 7) | 0xFBFF);
             retour&= ((this.gameBoard[5]<< 5) | 0xFEFF);
             break;
           default: // { 6, ? }
             retour&= ((this.gameBoard[y-2]<<12) | 0x0FFF);
             retour&= ((this.gameBoard[y-1]<< 7) | 0xFBFF);
             retour&= ((this.gameBoard[y  ]<< 5) | 0xFEFF);
             retour&= ((this.gameBoard[y+1]<< 3) | 0xFFBF);
             retour&= ((this.gameBoard[y+2]<< 1) | 0xFFE1);
         }// switch (y)
         break;

       case  7 :
         switch (y)
         { case 0 : // { 7, 0 }
             retour&= ((this.gameBoard[0]<< 6) | 0xFEFF);
             retour&= ((this.gameBoard[1]<< 4) | 0xFFBF);
             retour&= ((this.gameBoard[2]<< 2) | 0xFFE3);
             break;
           case 1 : // { 7, 1 }
             retour&= ((this.gameBoard[0]<< 8) | 0xFBFF);
             retour&= ((this.gameBoard[1]<< 6) | 0xFEFF);
             retour&= ((this.gameBoard[2]<< 4) | 0xFFBF);
             retour&= ((this.gameBoard[3]<< 2) | 0xFFE3);
             break;
           case 4 : // { 7, 4 }
             retour&= ((this.gameBoard[2]<<13) | 0x1FFF);
             retour&= ((this.gameBoard[3]<< 8) | 0xFBFF);
             retour&= ((this.gameBoard[4]<< 6) | 0xFEFF);
             retour&= ((this.gameBoard[5]<< 4) | 0xFFBF);
             break;
           case 5 : // { 7, 5 }
             retour&= ((this.gameBoard[3]<<13) | 0x1FFF);
             retour&= ((this.gameBoard[4]<< 8) | 0xFBFF);
             retour&= ((this.gameBoard[5]<< 6) | 0xFEFF);
             break;
           default: // { 7, ? }
             retour&= ((this.gameBoard[y-2]<<13) | 0x1FFF);
             retour&= ((this.gameBoard[y-1]<< 8) | 0xFBFF);
             retour&= ((this.gameBoard[y  ]<< 6) | 0xFEFF);
             retour&= ((this.gameBoard[y+1]<< 4) | 0xFFBF);
             retour&= ((this.gameBoard[y+2]<< 2) | 0xFFE3);
         }// switch (y)
         break;

       default :
         switch (y)
         { case 0 : // { ?, 0 }
             retour&= (( this.gameBoard[0]<<( 6-machin)) | 0xFEFF);
             retour&= (((this.gameBoard[1]>>machin)<<4 ) | 0xFFBF);
             retour&= (( this.gameBoard[0]<<( 9-machin)) | 0xFF7F);
             retour&= (( this.gameBoard[1]<<( 7-machin)) | 0xFFDF);
             retour&= (( this.gameBoard[2]>>(machin- 2)) | 0xFFE0);
             break;
           case 1 : // { ?, 1 }
             retour&= (( this.gameBoard[0]<<( 8-machin)) | 0xFBFF);
             retour&= (( this.gameBoard[1]<<( 6-machin)) | 0xFEFF);
             retour&= (((this.gameBoard[2]>>machin)<<4 ) | 0xFFBF);
             retour&= (( this.gameBoard[0]<<(11-machin)) | 0xFDFF);
             retour&= (( this.gameBoard[1]<<( 9-machin)) | 0xFF7F);
             retour&= (( this.gameBoard[2]<<( 7-machin)) | 0xFFDF);
             retour&= (( this.gameBoard[3]>>(machin- 2)) | 0xFFE0);
             break;
           case 4 : // { ?, 4 }
             retour&= (( this.gameBoard[2]<<(13-machin)) | 0x07FF);
             retour&= (( this.gameBoard[3]<<( 8-machin)) | 0xFBFF);
             retour&= (( this.gameBoard[4]<<( 6-machin)) | 0xFEFF);
             retour&= (((this.gameBoard[5]>>machin)<<4 ) | 0xFFBF);
             retour&= (( this.gameBoard[3]<<(11-machin)) | 0xFDFF);
             retour&= (( this.gameBoard[4]<<( 9-machin)) | 0xFF7F);
             retour&= (( this.gameBoard[5]<<( 7-machin)) | 0xFFDF);
             break;
           case 5 : // { ?, 5 }
             retour&= (( this.gameBoard[3]<<(13-machin)) | 0x07FF);
             retour&= (( this.gameBoard[4]<<( 8-machin)) | 0xFBFF);
             retour&= (( this.gameBoard[5]<<( 6-machin)) | 0xFEFF);
             retour&= (( this.gameBoard[4]<<(11-machin)) | 0xFDFF);
             retour&= (( this.gameBoard[5]<<( 9-machin)) | 0xFF7F);
             break;
           default: // { ?, ? }
             retour&= (( this.gameBoard[machin-2]<<(13-machin)) | 0x07FF);
             retour&= (( this.gameBoard[machin-1]<<( 8-machin)) | 0xFBFF);
             retour&= (( this.gameBoard[machin  ]<<( 6-machin)) | 0xFEFF);
             retour&= (((this.gameBoard[machin+1]>>machin)<<4 ) | 0xFFBF);
             retour&= (( this.gameBoard[machin-1]<<(11-machin)) | 0xFDFF);
             retour&= (( this.gameBoard[machin  ]<<( 9-machin)) | 0xFF7F);
             retour&= (( this.gameBoard[machin+1]<<( 7-machin)) | 0xFFDF);
             retour&= (( this.gameBoard[machin+2]>>(machin- 2)) | 0xFFE0);
         }// switch(y)
    }// switch (x)

   // on retablit l'etat des cases occupees par les joueurs
    this.switchPoint( this.positions[Game.PLAYER_ONE] );
    this.switchPoint( this.positions[Game.PLAYER_TWO] );

    return ((~retour) & masque);
  }


 /**
  * Recupere les positions des cases situees a un mouvement d'ecart
  *  du point p.
  *
  * @param p                position a considerer
  * @return                 mot binaire
  */
  public final int getLayer2(GamePoint p)
  { return (this.getLayer2(p.x, p.y)); }


 /**
  * Force le contexte de l'etat.
  *
  * @param context          nouveau contexte
  */
  final void setGameContext(int context)
  { this.gameContxt= context; }



 /* # methodes  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
 /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

 /**
  * Modification de l'etat de la case situee au point p, SANS modification
  *  de contexte.
  *
  * @param p                position a considerer
  */
  private final void switchPoint(GamePoint p)
  { this.gameBoard[p.y]^= (byte)(1 << (7-p.x)); }



 /**
  * Indique si le joueur player peut se deplacer, et donc encore jouer.
  *
  * @param playerId         identifiant du joueur
  * @return                 false si le joueur est immobilise
  */
  final boolean canPlay(int playerId)
  { return (0 != this.getLayer1(this.positions[playerId])); }


 /**
  * Indique si le coup suggere, ici un deplacement, est acceptable ou non.
  *
  * @param playerId         identifiant du joueur
  * @param x                abscisse du deplacement
  * @param y                ordonnee du deplacement
  * @return                 true si le deplacement est valable
  */
  final boolean isValidMove(int playerId, int x, int y)
  { boolean retour= false;
    try
    { retour= 0 == (gameBoard[y] & (byte)(1 << (7-x)))
           && 1 == this.positions[playerId & 0x01].distance(x, y)
           && !this.positions[Game.PLAYER_ONE].equals(x, y)
           && !this.positions[Game.PLAYER_TWO].equals(x, y) ;
    }
    catch (ArrayIndexOutOfBoundsException e)
    { /* endless tears, forever joy */ }
    return (retour);
  }


 /**
  * Indique si le coup suggere, ici un deplacement, est acceptable ou non.
  *
  * @param playerId         identifiant du joueur
  * @param p                position du deplacement
  * @return                 true si le deplacement est valable
  */
  final boolean isValidMove(int playerId, GamePoint p)
  { return (this.isValidMove(playerId, p.x, p.y)); }


 /**
  * Indique si le coup suggere, ici la destruction d'une case, est oui ou
  *  non acceptable.
  *
  * @param playerId         identifiant du joueur
  * @param x                abscisse du coup
  * @param y                ordonnee du coup
  * @return                 true si le coup est valable
  */
  final boolean isValidPush(int playerId, int x, int y)
  { boolean retour= false;
    try
    { retour= 0 == (gameBoard[y] & (byte)(1 << (7-x)))
           && !this.positions[Game.PLAYER_ONE].equals(x, y)
           && !this.positions[Game.PLAYER_TWO].equals(x, y) ;
    }
    catch (ArrayIndexOutOfBoundsException e)
    { /* to feel most every feeling, forever more */ }
    return (retour);
  }


 /**
  * Indique si le coup suggere, ici la destruction d'une case, est oui ou
  *  non acceptable.
  *
  * @param playerId         identifiant du joueur
  * @param p                position du coup
  * @return                 true si le coup est valable
  */
  final boolean isValidPush(int playerId, GamePoint p)
  { return (this.isValidPush(playerId, p.x, p.y)); }


 /**
  * Deplacement du joueur player vers la position (x,y), et mise a jour
  *  du contexte.
  *
  * @param playerId         identifiant du joueur
  * @param x                abscisse
  * @param y                ordonnee
  */
  final void move(int playerId, int x, int y)
  { int safeId= (playerId & 0x01);
    this.positions[safeId].move(x, y);
    this.gameContxt= Game.PLAYER_ONE_MOVE + safeId;
  }


 /**
  * Deplacement du joueur player vers le point p, et mise a jour du contexte.
  * 
  * @param playerId         identifiant du joueur
  * @param p                nouvelles coordonnes
  */
  final void move(int playerId, GamePoint p)
  { this.move(playerId, p.x, p.y); }


 /**
  * Destruction de la case a la position (x,y), et mise a jour du contexte.
  *
  * @param playerId         identifiant du joueur
  * @param x                abscisse
  * @param y                ordonnee
  */
  final void push(int playerId, int x, int y)
  { this.gameBoard[y]|= (byte)(1 << (7-x));
    this.gameContxt= Game.PLAYER_ONE_PUSH + (playerId & 0x01);
  }


 /**
  * Destruction de la case au point p, et mise a jour du contexte.
  * 
  * @param playerId         identifiant du joueur
  * @param p                case a detruire
  */
  final void push(int playerId, GamePoint p)
  { this.push(playerId, p.x, p.y); }


 /**
  * Remise a zero du jeu, toutes les cases du plateau redeviennent
  *  pratiquables, les joueurs sont places a leurs positions initiales
  *  respectives.
  */
  final void reset()
  { this.gameBoard[0]= (byte)0;
    this.gameBoard[1]= (byte)0;
    this.gameBoard[2]= (byte)0;
    this.gameBoard[3]= (byte)0;
    this.gameBoard[4]= (byte)0;
    this.gameBoard[5]= (byte)0;

    this.positions[Game.PLAYER_ONE].move(Game.POSITIONS[Game.PLAYER_ONE]);
    this.positions[Game.PLAYER_TWO].move(Game.POSITIONS[Game.PLAYER_TWO]);

    this.gameContxt= Game.NO_GAME_CONTEXT;
  }



}// class GameState


/* # End of File - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */